/*
 * Copyright (c) 2021, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "ti_msp_dl_config.h"
#include "mcf_configuration.h"
#include "mcf_i2c.h"

/*MCF_I2C_configure*/
unsigned long pre_eeprom_write_regs[EEPROM_ARRAY_SIZE] = {0};
unsigned long post_eeprom_write_regs[EEPROM_ARRAY_SIZE] = {0};
bool MCF_I2C_configure(void);

/*MCF_I2C_fault_read*/
unsigned long gate_driver_fault_status_regs;
unsigned long controller_fault_status_regs;
bool MCF_I2C_faultcheck(void);

unsigned long current_algorithm_state_regs;

/*
 * VM_VOLTAGE Register
 * Bit 31-0, address = 0x478
 * 32-bit value indicating DC bus voltage;
 * DC Bus Voltage (V) = VM_VOLTAGE * 60 / 2^27
 * */
unsigned long vm_voltage_regs;

/*
 * SPEED_REF_CLOSED_LOOP Register
 * Bit 31-0, address = 0x5CC
 * 32-bit value indicating reference for speed loop;
 * Speed reference in closed loop (Hz) = (SPEED_REF_CLOSED_LOOP/ 2^27) * MAX_SPEED
 * */
unsigned long speed_ref_close_loop_regs;

/*
 * SPEED_FDBK Register
 * Bit 31-0, address = 0x752
 * 32-bit value indicating estimated rotor speed; EstimatedSpeed (Hz) = (SPEED_FDBK / 227)* MAX_SPEED
 * */
unsigned long speed_fdbk_regs;
bool MCF_I2C_statuscheck(void);

/*User speed control*/
unsigned long SetMotorspeed = 0xBFFF0005;   //need to be initialized for the start up speed
unsigned long VALUE_read;
unsigned long SPEED_feedback = 0;
unsigned long VOLT_feedback = 0;

/*record the counter value for capture*/
volatile uint32_t gCaptureCnt;
/*flag indicate that PWM duty edge is captured*/
volatile bool gSynced;
/*flag indicate that speed is updated through PWM*/
volatile bool SPEEDUPDATE;
uint32_t gLoadValue;
uint32_t GETPWMDUTY;

/*flag used to mark whether need to re-burn the eeprom*/
bool write_needed = false;

/*record the counter value for system management*/
unsigned long COUNTER=0x00000000;


int main(void)
{
    SYSCFG_DL_init();

    /*Enable interrupts*/
    NVIC_EnableIRQ(I2C_INST_INT_IRQN);   //enable I2C interrupt
    NVIC_EnableIRQ(CAPTURE_0_INST_INT_IRQN);   //enable timer interrupt
    NVIC_EnableIRQ(TIMER_0_INST_INT_IRQN);

    /*I2C variable configuration*/
    gI2cControllerStatus = I2C_STATUS_IDLE;
    gTxLen = I2C_TX_PACKET_SIZE;
    gRxLen = I2C_RX_PACKET_SIZE;

    /*Set LED to indicate start of transfer*/
    //DL_GPIO_setPins(GPIO_LEDS_PORT, GPIO_LEDS_USER_LED_1_PIN);   //PA26 Red

    gLoadValue = DL_TimerG_getLoadValue(CAPTURE_0_INST);

    /*Initialize capture global states*/
    gSynced = false;
    SPEEDUPDATE = true;

    /*
     * Forcing timers to halt immediately to prevent timers getting out of sync
     * when code is halted
     */
   DL_TimerG_setCoreHaltBehavior(
       CAPTURE_0_INST, DL_TIMER_CORE_HALT_IMMEDIATE);
   DL_TimerG_startCounter(CAPTURE_0_INST);
   DL_TimerG_startCounter(TIMER_0_INST);

    if (false == MCF_I2C_configure())
    {
        __BKPT(0);
    } else {
        I2C_write(0xEC, 0x0000);   //set MCF speed Reference input = (DIGITAL_SPEED_CTRL/32768)*100%
        while(1) {;}
    }
}



void I2C_INST_IRQHandler(void)
{
    switch (DL_I2C_getPendingInterrupt(I2C_INST))
    {
        case DL_I2C_IIDX_CONTROLLER_RX_DONE:
            gI2cControllerStatus = I2C_STATUS_RX_COMPLETE;
            break;
        case DL_I2C_IIDX_CONTROLLER_TX_DONE:
            DL_I2C_disableInterrupt(
                I2C_INST, DL_I2C_INTERRUPT_CONTROLLER_TXFIFO_TRIGGER);
            gI2cControllerStatus = I2C_STATUS_TX_COMPLETE;
            break;
        case DL_I2C_IIDX_CONTROLLER_RXFIFO_TRIGGER:
            gI2cControllerStatus = I2C_STATUS_RX_INPROGRESS;
            /*Receive all bytes from target*/
            while (DL_I2C_isControllerRXFIFOEmpty(I2C_INST) != true)
            {
                if (gRxCount < gRxLen) {
                    gRxPacket[gRxCount++] =
                        DL_I2C_receiveControllerData(I2C_INST);
                } else {
                    /*Ignore and remove from FIFO if the buffer is full*/
                    DL_I2C_receiveControllerData(I2C_INST);
                }
            }
            break;
        case DL_I2C_IIDX_CONTROLLER_TXFIFO_TRIGGER:
            gI2cControllerStatus = I2C_STATUS_TX_INPROGRESS;
            /*Fill TX FIFO with next bytes to send*/
            if (gTxCount < I2C_TX_MAX_PACKET_SIZE)
            {
                gTxCount += DL_I2C_fillControllerTXFIFO(
                    I2C_INST, (uint8_t*) &gTxPacket[gTxCount], I2C_TX_MAX_PACKET_SIZE - gTxCount);
            }
            break;
        case DL_I2C_IIDX_CONTROLLER_ARBITRATION_LOST:
            break;
        case DL_I2C_IIDX_CONTROLLER_NACK:
            if ((gI2cControllerStatus == I2C_STATUS_RX_STARTED) ||
                (gI2cControllerStatus == I2C_STATUS_TX_STARTED))
            {
                /*NACK interrupt if I2C Target is disconnected*/
                gI2cControllerStatus = I2C_STATUS_ERROR;
            }
        default:
            break;
    }
}


/*Timer G to capture input PWM duty cycle*/
void CAPTURE_0_INST_IRQHandler(void)
{
    switch (DL_TimerG_getPendingInterrupt(CAPTURE_0_INST)) {
        case DL_TIMERG_IIDX_CC1_DN:
            /*If PWM signal is in (0,100) range this branch will detect the PWM duty cycle*/
            if (gSynced == true) {
                gCaptureCnt = DL_TimerG_getCaptureCompareValue(CAPTURE_0_INST, DL_TIMER_CC_1_INDEX);
            } else {
                gSynced = true;
            }
            /*Manual reload is needed to workaround timer capture limitation*/
            DL_TimerG_setTimerCount(CAPTURE_0_INST, gLoadValue);
            SPEEDUPDATE = true;
            break;
        case DL_TIMERG_IIDX_ZERO:
            /*If PWM signal is 0 or 100 detect GPIO level to identify the PWM duty*/
            gSynced = false;
            SPEEDUPDATE = true;
            break;
        default:
            break;
    }
}


/* Timer A to create 1ms time base for system management*/
void TIMER_0_INST_IRQHandler(void)
{
    switch (DL_TimerG_getPendingInterrupt(TIMER_0_INST)) {
        case DL_TIMER_IIDX_ZERO:
            COUNTER++;   //1ms time base counter
            /*read DC bus voltage*/
            I2C_read(VM_VOLTAGE, &VOLT_feedback);
            VOLT_feedback = VOLT_feedback*60>>27;
            /*update FG speed*/
            if (gSynced)
            {
               SPEED_feedback = gLoadValue*2/(gLoadValue - gCaptureCnt);
            }
            break;
        default:
            break;
    }
}


bool MCF_I2C_configure(void)
{
    unsigned int i;
    bool prog_error = false;   //false

    ///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    /*Read EEPROM*/
    /* Set the EEPROM_READ bit in the DEV_CTRL register (0xEA located in RAM
     * This reads the content of the EEPROM memory into the shadow registers on MCF board
     */
    if (!I2C_write(0xEA, 0x40000000)) { //1st
    /* 7.6.1.2 EEPROM Read，
     * Write 0x40000000 into register 0x0000EA to read the EEPROM data into the shadow registers(0x000080-0x0000AE).
     */
     prog_error = true;   //true: error occurs
    }
    /*delay at least 100ms for copy operation from EEPROM to Shadow-register to complete p70*/
    delay_cycles(3200000);

    /*Read the initial EEPROM registers into pre_eeprom_write_regs array*/
    for (i = 0; i < EEPROM_ARRAY_SIZE; i++)
    {
        if(!I2C_read(eeprom_regmap[i][0], &(pre_eeprom_write_regs[i])))   //2nd
        /*read the value of 0x000080-0x0000AE registers to pre_eeprom_write_regs[i]*/
        {
            prog_error = true;
            break;
        }
        delay_cycles(6400);   //200us
    }
    delay_cycles(1600000);

    ///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    /*Write EEPROM*/
    /*Write the user configured EEPROM registers to the shadow register*/
    for (i = 0; i < EEPROM_ARRAY_SIZE; i++)
    {
        /* very important code which can save mcf burning life，
         * Only write if the EEPROM value differs from the user configuration，Write EEPROM if necessary
         */
        if (pre_eeprom_write_regs[i] != eeprom_regmap[i][1])
        {
            /*Only write if the value in EEPROM is different from the target value*/
            if (!I2C_write(eeprom_regmap[i][0], eeprom_regmap[i][1]))   //3rd
            {
                prog_error = true;
                break;
            }
            delay_cycles(6400);
            write_needed = true;   //mark need to be written in
        }
    }
    delay_cycles(3200000);

    /* Set the EEPROM_WRITE bit along with EEPROM_WRITE_ACCESS_KEY in the DEV_CTRL register (0xEA located in RAM)
     * This writes the content of the shadow registers into the EERPOM memory
     */
    /*7.6.1.1 EEPROM Write*/
    if (write_needed)
    {
        if (!I2C_write(0xEA, 0x8A500000)) {//4th
            prog_error = true;
        }
        delay_cycles(6400000);
    }
    // delay at least 100ms for copy operation from Shadow-register to EEPROM to complete

    ///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    /*Read EEPROM*/
    /* Set the EEPROM_READ bit in the DEV_CTRL register (0xEA located in RAM
     * This reads the content of the EEPROM memory back into the shadow registers
     */
    if (write_needed)
    {
        if (!I2C_write(0xEA, 0x40000000)) {   //5th
            prog_error = true;
        }
        delay_cycles(6400000); // delay at least 100ms for copy operation from EEPROM to Shadow-register to complete

        /* Read back the EEPROM registers into post_eeprom_write_regs array */
        for (i = 0; i < EEPROM_ARRAY_SIZE; i++)
        {
            if(!I2C_read(eeprom_regmap[i][0], &(post_eeprom_write_regs[i])))   //6th
            {
                prog_error = true;
                break;
            }
            delay_cycles(12800);
            /*Verify that the written value matches the target*/
            if (post_eeprom_write_regs[i] != eeprom_regmap[i][1])
            {
                prog_error = true;
                break;
            }
        }
    }
    if(prog_error) return false;
    else return true;
}


bool MCF_I2C_faultcheck(void)
{
    bool prog_error = false;   //false
    /*check MCF GATE_DRIVER_FAULT_STATUS Register*/
    if(!I2C_read(0xE0, &(gate_driver_fault_status_regs)))
    {
        prog_error = true;
    }
    /*check MCF CONTROLLER_FAULT_STATUS Register*/
    if(!I2C_read(0xE2, &(controller_fault_status_regs)))
    {
        prog_error = true;
    }
    if(prog_error) return false;
    else return true;
}


bool MCF_I2C_statuscheck(void)
{
    bool prog_error = false;   //false
    /*check MCF ALGORITHM_STATE Register*/
    if(!I2C_read(0x210, &(current_algorithm_state_regs)))
    {
        prog_error = true;
    }
    /*check MCF VM_VOLTAGE Register*/
    if(!I2C_read(0x478, &(vm_voltage_regs)))
    {
        prog_error = true;
    }
    /*check MCF SPEED_REF_CLOSED_LOOP Register*/
    if(!I2C_read(0x5CC, &(speed_ref_close_loop_regs)))
    {
        prog_error = true;
    }
    /*check MCF SPEED_FDBK Register*/
    if(!I2C_read(0x752, &(speed_fdbk_regs)))
    {
        prog_error = true;
    }
    if(prog_error) return false;
    else return true;
}
